<?php
namespace WhatsBoost;

if (!defined('ABSPATH')) { exit; }

final class Admin {

    /** Basename del archivo principal del plugin (carpeta/archivo.php) */
    private $plugin_basename;

    public function __construct() {
        // Detecta el archivo principal del plugin para que los enlaces solo se apliquen a este plugin.
        if (defined('WHATSBOOST_FILE')) {
            $this->plugin_basename = plugin_basename(WHATSBOOST_FILE);
        } else {
            // Fallback: intenta adivinar que el archivo principal se llama "whatsboost.php" en la raíz.
            $root = rtrim(dirname(__DIR__), '/\\');
            $guess = $root . '/whatsboost.php';
            $this->plugin_basename = plugin_basename($guess);
        }
    }

    public function hooks(): void {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_init', [$this, 'maybe_redirect_after_activation']);
        add_action('admin_init', [$this, 'add_privacy_policy_content']);
        add_action('admin_notices', [$this, 'license_notice']);

        // Refrescar estado de cuenta periódicamente y aviso global si falta
        add_action('admin_init', [$this, 'maybe_refresh_account_state']);
        add_action('admin_notices', [$this, 'account_notice'], 1);
        // Aviso de éxito persistente cuando todo está correcto
        add_action('admin_notices', [$this, 'success_notice'], 9);
        // Forzar re-chequeo inmediato al abrir la página de ajustes de WhatsBoost
        add_action('load-toplevel_page_whatsboost', [$this, 'force_refresh_account_once']);

        add_action('admin_enqueue_scripts', [$this, 'assets']);
        add_action('wp_ajax_whatsboost_verify_key', [$this, 'ajax_verify_key']);
        add_action('wp_ajax_whatsboost_get_accounts', [$this, 'ajax_get_accounts']);
        add_action('wp_ajax_whatsboost_set_account', [$this, 'ajax_set_account']);

        // ► Enlace "Documentación" en la fila del plugin (al lado de "By ALMC")
        add_filter('plugin_row_meta', [$this, 'add_docs_meta_link'], 10, 2);

        // ► (Opcional) Enlace "Documentación" también en los action links (Settings | Deactivate)
        add_filter('plugin_action_links_' . $this->plugin_basename, [$this, 'add_action_link_docs']);
    }

    public function menu(): void {
        // Menú principal: WhatsBoost (dashboard simple)
        add_menu_page(
            __('WhatsBoost', 'whatsboost'),
            __('WhatsBoost', 'whatsboost'),
            'manage_options',
            'whatsboost',
            [$this, 'dashboard_page'],
            WHATSBOOST_URL . 'assets/img/favicon.png'
        );

        // Hacemos visible el mismo como primera opción del submenú
        add_submenu_page(
            'whatsboost',
            __('WhatsBoost', 'whatsboost'),
            __('WhatsBoost', 'whatsboost'),
            'manage_options',
            'whatsboost',
            [$this, 'dashboard_page']
        );

        // Subpágina de Documentación (independiente del dashboard)
        add_submenu_page(
            'whatsboost',
            __('Documentación', 'whatsboost'),
            __('Documentación', 'whatsboost'),
            'manage_options',
            'whatsboost-docs',
            [$this, 'render_page']
        );

        // (Opcional) Página de configuración de estados de WooCommerce
        add_submenu_page(
            'whatsboost',
            __('Estados de pedido', 'whatsboost'),
            __('Estados de pedido', 'whatsboost'),
            'manage_options',
            'whatsboost-bienvenido',
            [$this, 'welcome_page']
        );
    }

    public function register_settings(): void {

        register_setting('whatsboost_group', 'whatsboost_license_key', [
            'type' => 'string',
            'sanitize_callback' => function($value){ return preg_replace('/\s+/', '', sanitize_text_field($value)); },
            'default' => '',
        ]);

        // Guardado de estados habilitados por estado de pedido
        register_setting('whatsboost_status_group', 'whatsboost_status_enabled', [
            'type' => 'array',
            'sanitize_callback' => function($value){
                if (!is_array($value)) { return []; }
                $out = [];
                foreach ($value as $k => $v) {
                    $key = sanitize_key((string)$k);
                    // Guardamos 1 para marcado
                    $out[$key] = 1;
                }
                return $out;
            },
            'default' => [],
        ]);

        // Guardado de texto por estado de pedido
        register_setting('whatsboost_status_group', 'whatsboost_status_text', [
            'type' => 'array',
            'sanitize_callback' => function($value){
                if (!is_array($value)) { return []; }
                $out = [];
                foreach ($value as $k => $v) {
                    $key = sanitize_key((string)$k);
                    $out[$key] = sanitize_textarea_field((string)$v);
                }
                return $out;
            },
            'default' => [],
        ]);

        add_settings_section(
            'whatsboost_section',
            __('Configuración de la licencia', 'whatsboost'),
            function(){ echo '<p>' . esc_html__('Pega tu clave para activar el plugin.', 'whatsboost') . '</p>'; },
            'whatsboost'
        );

        add_settings_field(
            'whatsboost_license_key',
            __('Clave de licencia', 'whatsboost'),
            [$this, 'field_license_key'],
            'whatsboost',
            'whatsboost_section'
        );
    }

    public function field_license_key(): void {
        $key = get_option('whatsboost_license_key', '');
        ?>
        <label for="whatsboost_license_key" class="screen-reader-text"><?php esc_html_e('Clave de licencia', 'whatsboost'); ?></label>
        <input id="whatsboost_license_key" aria-label="<?php echo esc_attr__('Clave de licencia', 'whatsboost'); ?>" type="text" name="whatsboost_license_key" value="<?php echo esc_attr($key); ?>" class="regular-text" placeholder="XXXXXXXXXXXXXX">
        <p class="description"><?php esc_html_e('Pega aquí tu clave de WhatsBoost.', 'whatsboost'); ?></p>
        <?php
    }

    public function assets($hook): void {
        // Cargamos los assets en cualquiera de las páginas del plugin (dashboard, docs, bienvenida)
        if (false === strpos((string) $hook, 'whatsboost')) { return; }
        wp_enqueue_style('whatsboost-admin', WHATSBOOST_URL . 'assets/css/admin.css', [], WHATSBOOST_VERSION);
        wp_enqueue_script('whatsboost-admin', WHATSBOOST_URL . 'assets/js/admin.js', ['jquery'], WHATSBOOST_VERSION, true);
        wp_localize_script('whatsboost-admin', 'WhatsBoostVerify', [
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce'   => wp_create_nonce('whatsboost_verify'),
                'welcomeUrl' => admin_url('admin.php?page=whatsboost-bienvenido'),
                'settingsUrl' => admin_url('admin.php?page=whatsboost'),
                'locale'  => (function_exists('determine_locale') ? determine_locale() : get_locale()),
                // Mostrar SIEMPRE el valor de la opción viva, no la constante
                'currentAccount' => (string) get_option('whatsboost_account', ''),
                'currentVerified' => (int) get_option('whatsboost_license_verified', 0),
                'currentKey'      => (string) get_option('whatsboost_license_key', ''),
                // Mensajes traducibles para el JS
                'messages' => [
                'genericError'         => __('Ha ocurrido un error.', 'whatsboost'),
                'genericSuccess'       => __('Operación realizada.', 'whatsboost'),
                'invalidKey'           => __('Clave no válida. Intenta con otra clave o vuelve a copiarla.', 'whatsboost'),
                'verifying'            => __('Verificando...', 'whatsboost'),
                'verified'             => __('Clave verificada', 'whatsboost'),
                'verifiedMsg'          => __('Clave verificada.', 'whatsboost'),
                'accountPreviewLabel'  => __('ID de cuenta detectado:', 'whatsboost'),
                'noAccountPreview'     => __('No hay ID de cuenta aún. Verifica tu clave para obtenerlo automáticamente.', 'whatsboost'),
                'noAccountAfterVerify' => __('Tu clave es válida, pero no hemos recibido el identificador de tu cuenta (el número de WhatsApp desde el que se enviarán los mensajes). Conéctalo en WhatsBoost y vuelve a verificar.', 'whatsboost'),
                'successUseConfigured' => __('Clave verificada. Usaremos tu número/cuenta configurado.', 'whatsboost'),
                'needPhoneWithAccount' => __('Te falta poner tu número de teléfono para enviar WhatsApps. Copia esta clave y pégala en whatsboost.php (WHATSBOOST_ACCOUNT):', 'whatsboost'),
                'mismatchedAccount'    => __('La clave pertenece a otro número/cuenta. Actualiza whatsboost.php con esta clave para enviar desde tu número:', 'whatsboost'),
                'allGood'              => __('Todo correcto. Enviaremos WhatsApps desde tu número.', 'whatsboost'),
                'invalidLicense'       => __('Clave de licencia no válida. Inténtalo de nuevo.', 'whatsboost'),
                'networkError'         => __('Error de red.', 'whatsboost'),
                'accountNoticePrefix'  => __('WhatsBoost: Necesitas poner tu número de teléfono para que se envíen los WhatsApps desde tu teléfono.', 'whatsboost'),
                'dismissLabel'         => __('Dismiss', 'whatsboost'),
                'openSettings'         => __('Abrir ajustes de WhatsBoost', 'whatsboost'),
                "replaceKeyConfirm"    => __("Vas a reemplazar la clave guardada por la nueva.\n\nSi continúas, la clave anterior se sustituirá por esta.\n\n¿Quieres continuar?", 'whatsboost'),
                // Added for table and statuses in JS (translatable)
                'colTelefono'          => __('Teléfono', 'whatsboost'),
                'colIdCuenta'          => __('ID de cuenta', 'whatsboost'),
                'colEstado'            => __('Estado', 'whatsboost'),
                'statusWorks'          => __('Funciona', 'whatsboost'),
                'statusNoWorks'        => __('No funciona', 'whatsboost'),
                'noWorkingAccounts'    => __('No hay cuentas conectadas. Solo aparecen aquí las que funcionan.', 'whatsboost'),
            ],
        ]);
    }

    public function render_page(): void {
        if (!current_user_can('manage_options')) { return; }
        $file = WHATSBOOST_DIR . 'includes/admin-views/documentos.php';
        if (file_exists($file)) {
            include $file;
        } else {
            echo '<div class="wrap"><h1>' . esc_html__('Documentación', 'whatsboost') . '</h1>';
            echo '<p>' . esc_html__('No se encontró la vista de documentación.', 'whatsboost') . '</p></div>';
        }
    }

    public function dashboard_page(): void {
        if (!current_user_can('manage_options')) { return; }

        // Si al cargar la página la clave está vacía, forzar la marca de verificación a 0
        $key = trim((string) get_option('whatsboost_license_key', ''));
        if ($key === '') {
            update_option('whatsboost_license_verified', 0, false);
        }

        // Verificar/actualizar el estado de la clave al entrar a la página (sin esperar al cron/transient)
        if (method_exists($this, 'force_refresh_account_once')) {
            $this->force_refresh_account_once();
        }

        $file = WHATSBOOST_DIR . 'includes/admin-views/settings-page.php';
        if (file_exists($file)) {
            include $file;
            return;
        }

        // Si la vista no existe, no renderizamos HTML desde la clase (evitar duplicidad).
        return;
    }

    public function welcome_page(): void {
        if (!current_user_can('manage_options')) { return; }

        $key = get_option('whatsboost_license_key', '');
        if ($key === '') {
            wp_safe_redirect(admin_url('admin.php?page=whatsboost'));
            exit;
        }
        $file = WHATSBOOST_DIR . 'includes/admin-views/welcome-page.php';
        if (file_exists($file)) {
            include $file;
        } else {
            echo '<div class="wrap"><h1>' . esc_html__('Configuración completa', 'whatsboost') . '</h1>';
            echo '<p>' . esc_html__('Tu clave ha sido verificada. ¡Todo listo!', 'whatsboost') . '</p></div>';
        }
    }

    public function maybe_redirect_after_activation(): void {
        // Implementar si hace falta redirigir tras activar el plugin.
    }

    public function license_notice(): void {
        // Mostrar un aviso global en el admin si la clave está vacía o no es válida (verified != 1).
        if (!current_user_can('manage_options')) { return; }

        $key = trim((string) get_option('whatsboost_license_key', ''));
        $verified = (int) get_option('whatsboost_license_verified', 0);
        $should_show = ($key === '' || $verified !== 1);
        if (!$should_show) { return; }

        $user_id = get_current_user_id();
        // Permitir descartar SOLO cuando el problema es que no hay clave; si la clave no es válida, forzamos mostrar
        if ($key === '' && get_user_meta($user_id, 'whatsboost_dismiss_no_key_notice', true)) { return; }

        // Procesar opción de descartar mediante nonce (si viene en la URL)
        if ($key === '' && isset($_GET['whatsboost_dismiss_no_key'])) {
            if (isset($_REQUEST['_wpnonce']) && check_admin_referer('whatsboost_dismiss_no_key')) {
                update_user_meta($user_id, 'whatsboost_dismiss_no_key', 1);
                // Redirigir para limpiar la query y evitar repost
                $redirect = remove_query_arg(array('whatsboost_dismiss_no_key', '_wpnonce'));
                wp_safe_redirect($redirect);
                exit;
            }
        }

        $settings_url = admin_url('admin.php?page=whatsboost');
        $docs_url = admin_url('admin.php?page=whatsboost-docs');

        // Mensaje amigable que indica la necesidad de configurar/verificar la clave y cómo conseguirla.
        echo '<div id="whatsboost-no-key-notice" class="notice notice-warning is-dismissible">'
            . '<p style="font-size:14px; margin:0;">'
            . '<strong>' . esc_html__('WhatsBoost', 'whatsboost') . ':</strong> '
            . esc_html__('Para empezar a enviar WhatsApps necesitas introducir tu clave de licencia de WhatsBoost y asociar tu número. Es rápido y seguro — la obtienes en pocos pasos desde la documentación o la sección de ajustes.', 'whatsboost')
            . '</p>'
            . '<p style="margin-top:8px;">'
            . '<a class="button button-primary" href="' . esc_url($settings_url) . '">' . esc_html__('Abrir ajustes de WhatsBoost', 'whatsboost') . '</a>'
            . '<a class="button" style="margin-left:8px;" href="' . esc_url($docs_url) . '">' . esc_html__('Cómo conseguir la clave', 'whatsboost') . '</a>'
            . '</p>'
            . '</div>';
    }

    /**
     * Añade contenido a la política de privacidad de WordPress sobre el uso de datos del plugin.
     */
    public function add_privacy_policy_content(): void {
        if (!function_exists('wp_add_privacy_policy_content')) { return; }

        $content = '';
        $content .= '<p><strong>WhatsBoost</strong> ' . esc_html__('realiza solicitudes externas solo cuando pegas y verificas tu clave o cuando esta ya ha sido verificada, para comprobar el estado de la cuenta asociada.', 'whatsboost') . '</p>';
        $content .= '<ul>';
        $content .= '<li>' . esc_html__('Se envía tu clave (secreta) al servicio de WhatsBoost para validar la suscripción y recuperar el identificador de la cuenta (unique).', 'whatsboost') . '</li>';
        $content .= '<li>' . esc_html__('El plugin almacena en la base de datos de WordPress: la clave, una marca de verificación, el identificador de cuenta y un transient de control de comprobaciones.', 'whatsboost') . '</li>';
        $content .= '<li>' . esc_html__('En la desinstalación, el plugin elimina todas estas opciones y transients.', 'whatsboost') . '</li>';
        $content .= '</ul>';
        $content .= '<p>' . esc_html__('No se recopilan otros datos personales desde WordPress ni se envían datos del cliente/pedidos a menos que configures y utilices el servicio de WhatsBoost para enviar mensajes, en cuyo caso se envían únicamente los datos necesarios para el mensaje.', 'whatsboost') . '</p>';

        wp_add_privacy_policy_content(__('WhatsBoost', 'whatsboost'), wp_kses_post($content));
    }

    /**
     * Comprueba contra la API si sigue habiendo una cuenta (unique) válida para la clave.
     * Si ya no existe, borra la opción local para forzar el aviso global.
     * Se limita con un transient para no llamar en cada carga (p.ej. cada 5 minutos).
     */
    public function maybe_refresh_account_state(): void {
        if (!current_user_can('manage_options')) { return; }

        $key = trim((string) get_option('whatsboost_license_key', ''));
        if ($key === '') { return; }
        $verified = (int) get_option('whatsboost_license_verified', 0);
        if ($verified !== 1) { return; }

        // No consultar más de una vez cada 5 minutos
        $flag = get_transient('whatsboost_account_check_ts');
        if ($flag) { return; }

        set_transient('whatsboost_account_check_ts', time(), 5 * MINUTE_IN_SECONDS);

        $endpoint = apply_filters('whatsboost_accounts_endpoint', 'https://whatsboost.net/api/get/wa.accounts', $key);
        $url = add_query_arg(['secret' => $key, 'limit' => 100, 'page' => 1], $endpoint);
        $args = [ 'headers' => [ 'Accept' => 'application/json' ], 'timeout' => 15 ];

        $res  = wp_remote_get($url, $args);
        if (is_wp_error($res)) { return; }

        $code = (int) wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);
        if (!($code >= 200 && $code < 300) || !is_string($body) || $body === '') { return; }

        $json = json_decode($body, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($json)) { return; }

        $data = isset($json['data']) && is_array($json['data']) ? $json['data'] : [];

        // Normalizar lista completa: id + status
        $list = [];
        foreach ($data as $row) {
            if (!is_array($row)) { continue; }
            $id = '';
            foreach (['unique','account','account_id','accountId'] as $k) {
                if (isset($row[$k]) && is_string($row[$k]) && trim($row[$k]) !== '') { $id = trim((string)$row[$k]); break; }
            }
            if ($id === '') { continue; }
            $status = '';
            if (isset($row['status'])) { $status = is_string($row['status']) ? trim($row['status']) : ($row['status'] ? 'connected' : 'disconnected'); }
            elseif (isset($row['state'])) { $status = is_string($row['state']) ? trim($row['state']) : ''; }
            elseif (isset($row['connected'])) { $status = $row['connected'] ? 'connected' : 'disconnected'; }
            $list[] = [ 'id' => $id, 'status' => strtolower((string)$status) ];
        }

        $saved = trim((string) get_option('whatsboost_account', ''));

        // Si ya hay selección, NO sobrescribirla salvo que: no exista o esté desconectada y haya alguna conectada
        if ($saved !== '') {
            $exists = false; $savedStatus = '';
            foreach ($list as $it) {
                if ($it['id'] === $saved) { $exists = true; $savedStatus = $it['status']; break; }
            }
            if (!$exists) {
                // Mover a la primera conectada disponible si la guardada ya no existe
                foreach ($list as $it) { if ($it['status'] === 'connected') { update_option('whatsboost_account', $it['id'], false); return; } }
                // Si no hay conectadas, limpiar para forzar al usuario a vincular
                delete_option('whatsboost_account');
                return;
            }
            if ($savedStatus !== 'connected') {
                foreach ($list as $it) { if ($it['status'] === 'connected') { update_option('whatsboost_account', $it['id'], false); return; } }
                // Si no hay conectadas, limpiar
                delete_option('whatsboost_account');
            }
            return; // Respetar la selección existente
        }

        // Si no hay selección guardada, inicializar con la primera conectada; si no, la primera de la lista
        if (!empty($list)) {
            foreach ($list as $it) { if ($it['status'] === 'connected') { update_option('whatsboost_account', $it['id'], false); return; } }
            update_option('whatsboost_account', $list[0]['id'], false);
        }
    }

    /**
     * Aviso global cuando falta el ID de cuenta (unique) para enviar WhatsApps.
     */
    public function account_notice(): void {
        if (!current_user_can('manage_options')) { return; }

        $key = trim((string) get_option('whatsboost_license_key', ''));
        if ($key === '') { return; }

        $account = trim((string) get_option('whatsboost_account', ''));
        if ($account !== '') { return; }

        // Mostrar este aviso solo si la clave ha sido verificada correctamente
        $verified = (int) get_option('whatsboost_license_verified', 0);
        if ($verified !== 1) { return; }

        $settings_url = admin_url('admin.php?page=whatsboost');

        echo '<div id="whatsboost-account-missing-notice" class="notice notice-error" style="border-left-color:#c00">'
            . '<p style="font-size:14px;">'
            . esc_html__('WhatsBoost: Necesitas poner tu número de teléfono para que se envíen los WhatsApps desde tu teléfono.', 'whatsboost')
            . '</p>'
            . '<p style="margin-top:6px;">'
            . '<a class="button button-primary" href="' . esc_url($settings_url) . '">' . esc_html__('Abrir ajustes de WhatsBoost', 'whatsboost') . '</a>'
            . '</p>'
            . '</div>';
    }

    /**
     * Aviso de éxito persistente cuando hay clave y cuenta configuradas.
     * Se muestra en las páginas del plugin para confirmar que todo está correcto.
     */
    public function success_notice(): void {
        if (!current_user_can('manage_options')) { return; }

        // Mostrar solo en pantallas del plugin para no saturar el admin
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if ($screen && is_object($screen)) {
            $id = (string) $screen->id;
            // Mostrar SOLO en la página de ajustes de WhatsBoost (no en Docs ni Bienvenido)
            $allowed = [
                'toplevel_page_whatsboost',
                'whatsboost_page_whatsboost',
            ];
            if (!in_array($id, $allowed, true)) { return; }
        }

        $key = trim((string) get_option('whatsboost_license_key', ''));
        if ($key === '') { return; }

        $account = trim((string) get_option('whatsboost_account', ''));
        if ($account === '') { return; }

        echo '<div class="notice notice-success is-dismissible">'
            . '<p>' . esc_html__('Todo correcto. Enviaremos WhatsApps desde tu número.', 'whatsboost') . '</p>'
            . '</div>';
    }

    /**
     * Forzar refresco inmediato del estado de la cuenta al abrir la página de ajustes.
     * Elimina el transient anti-rebote y ejecuta la comprobación en esta misma carga.
     */
    public function force_refresh_account_once(): void {
        if (!current_user_can('manage_options')) { return; }
        delete_transient('whatsboost_account_check_ts');
        if (method_exists($this, 'maybe_refresh_account_state')) {
            $this->maybe_refresh_account_state();
        }
    }

    public function ajax_verify_key(): void {
        // Verificación de clave y listado de cuentas
        check_ajax_referer('whatsboost_verify', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('No tienes permisos suficientes para realizar esta acción.', 'whatsboost')], 403);
        }

        // Tomar la clave enviada por POST (no requerimos guardar previamente)
        $raw = isset($_POST['key']) ? (string) $_POST['key'] : '';
        $key = preg_replace('/\s+/', '', sanitize_text_field($raw));

        if ($key === '') {
            // Si la clave viene vacía, marcar como no verificada y persistir estado
            update_option('whatsboost_license_key', $key, false);
            update_option('whatsboost_license_verified', 0, false);
            update_option('whatsboost_account', '', false);
            wp_send_json_error([
                'message' => __('Clave no válida. Intenta con otra clave o vuelve a copiarla.', 'whatsboost'),
            ], 400);
        }

        // Endpoint de verificación según documentación: GET /api/get/subscription?secret=KEY
        $default_endpoint = 'https://whatsboost.net/api/get/subscription';
        $endpoint = defined('WHATSBOOST_VERIFY_URL') ? constant('WHATSBOOST_VERIFY_URL') : $default_endpoint;
        /** Permitir personalizar via filtro */
        $endpoint = apply_filters('whatsboost_verify_endpoint', $endpoint, $key);

        // Construir URL con ?secret=...
        $url = add_query_arg(['secret' => $key], $endpoint);

        // Hacer la llamada remota (sin Authorization Bearer)
        $args = [
            'headers' => [
                'Accept' => 'application/json',
            ],
            'timeout' => 12,
        ];

        $res  = wp_remote_get($url, $args);
        $code = is_wp_error($res) ? 0 : (int) wp_remote_retrieve_response_code($res);
        $body = is_wp_error($res) ? '' : wp_remote_retrieve_body($res);
        $json = null;

        if (is_wp_error($res)) {
            // Persistir siempre la clave; marcar no verificada y limpiar account
            update_option('whatsboost_license_key', $key, false);
            update_option('whatsboost_license_verified', 0, false);
            update_option('whatsboost_account', '', false);
            wp_send_json_error([
                'message' => __('No se pudo conectar al servidor de verificación. Inténtalo más tarde.', 'whatsboost'),
                'error'   => $res->get_error_message(),
            ], 500);
        }

        if (is_string($body) && $body !== '') {
            $decoded = json_decode($body, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $json = $decoded;
            }
        }

        // Si la respuesta NO es 2xx, tratamos como error inmediatamente
        $message = '';
        if ($json && isset($json['message']) && is_string($json['message'])) {
            $message = (string) $json['message'];
        }
        if (!($code >= 200 && $code < 300)) {
            if ($code === 401 || $code === 403) {
                $message = __('Clave de licencia no válida. Inténtalo de nuevo.', 'whatsboost');
            } elseif (!$message) {
                $message = __('No se pudo validar la clave. Inténtalo de nuevo más tarde.', 'whatsboost');
            }
            if (is_string($message) && stripos($message, 'Invalid API secret supplied') !== false) {
                $message = __('Clave de licencia no válida. Inténtalo de nuevo.', 'whatsboost');
            }
            // Persistir SIEMPRE la clave; marcar no verificada y limpiar account en errores
            update_option('whatsboost_license_key', $key, false);
            update_option('whatsboost_license_verified', 0, false);
            update_option('whatsboost_account', '', false);
            wp_send_json_error([
                'message' => $message,
                'status'  => $code,
            ], ($code >= 400 ? $code : 400));
        }

        // Éxito provisional (HTTP 2xx): pero validamos el contenido
        $message = __('Clave verificada.', 'whatsboost');

        // Señales de respuesta válida de la API (sin exigir account todavía)
        $api_status = is_array($json) && array_key_exists('status', $json) ? (int) $json['status'] : null;
        $api_message = is_array($json) && isset($json['message']) && is_string($json['message']) ? trim($json['message']) : '';
        $looks_subscription_ok = ($api_status === 200) || (stripos($api_message, 'subscription') !== false);

        // Detectar mensajes de error incluso con 200 OK, acotado a mensajes conocidos
        if ($json && isset($json['message']) && is_string($json['message'])) {
            $msg = trim((string) $json['message']);
            $knownBad = [
                'Invalid API secret supplied',
                'Invalid license key',
            ];
            if (in_array($msg, $knownBad, true)) {
                // Persistir clave y limpiar estado al ser inválida
                update_option('whatsboost_license_key', $key, false);
                update_option('whatsboost_license_verified', 0, false);
                update_option('whatsboost_account', '', false);
                wp_send_json_error([
                    'message' => __('Clave de licencia no válida. Inténtalo de nuevo.', 'whatsboost'),
                    'status'  => $code,
                ], 400);
            }
        }

        // Intentar extraer un identificador de cuenta válido
        $account_from_api = '';
        if ($json) {
            $candidates = [
                'unique', 'account', 'account_id', 'accountId',
                'data.subscription.account', 'data.subscription.account_id',
                'data.account', 'data.account_id',
            ];
            foreach ($candidates as $path) {
                $val = whatsboost_array_get_dot($json, $path);
                if (is_string($val) && trim($val) !== '') { $account_from_api = trim((string)$val); break; }
            }
        }

        // Consultar /wa.accounts para obtener todas las cuentas y, si falta, completar el ID
        $wa_accounts_list = [];
        {
            $wa_accounts_endpoint = apply_filters('whatsboost_accounts_endpoint', 'https://whatsboost.net/api/get/wa.accounts', $key);
            $wa_url = add_query_arg(['secret' => $key, 'limit' => 100, 'page' => 1], $wa_accounts_endpoint);
            $wa_res = wp_remote_get($wa_url, [ 'headers' => [ 'Accept' => 'application/json', 'Content-Type' => 'application/json' ], 'timeout' => 20 ]);
            $wa_code = is_wp_error($wa_res) ? 0 : (int) wp_remote_retrieve_response_code($wa_res);
            if (!is_wp_error($wa_res) && $wa_code >= 200 && $wa_code < 300) {
                $wa_body = wp_remote_retrieve_body($wa_res);
                $wa_json = json_decode($wa_body, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($wa_json)) {
                    $data = isset($wa_json['data']) && is_array($wa_json['data']) ? $wa_json['data'] : [];
                    foreach ($data as $row) {
                        if (!is_array($row)) { continue; }
                        $id = '';
                        foreach (['unique','account','account_id','accountId'] as $k) {
                            if (isset($row[$k]) && is_string($row[$k]) && trim($row[$k]) !== '') { $id = trim((string)$row[$k]); break; }
                        }
                        $phone = '';
                        foreach (['phone','mobile','msisdn','telefono'] as $k) {
                            if (isset($row[$k]) && (is_string($row[$k]) || is_numeric($row[$k]))) { $phone = trim((string)$row[$k]); break; }
                        }
                        $status = '';
                        if (isset($row['status'])) { $status = is_string($row['status']) ? trim($row['status']) : ($row['status'] ? 'connected' : 'disconnected'); }
                        elseif (isset($row['state'])) { $status = is_string($row['state']) ? trim($row['state']) : ''; }
                        elseif (isset($row['connected'])) { $status = $row['connected'] ? 'connected' : 'disconnected'; }
                        $wa_accounts_list[] = [ 'id' => $id, 'phone' => $phone, 'status' => $status ];
                        if ($account_from_api === '' && $id !== '') { $account_from_api = $id; }
                    }
                }
            }
        }

        // Si no hay account pero la suscripción parece correcta, permitimos verificar
        if ($account_from_api === '' && $looks_subscription_ok) {
            update_option('whatsboost_license_key', $key, false);
            update_option('whatsboost_license_verified', 1, false);
            // Limpiar cualquier cuenta previa: esta clave válida aún no tiene cuenta asociada
            update_option('whatsboost_account', '', false);
            delete_transient('whatsboost_account_check_ts');

            $defined_account = (string) get_option('whatsboost_account', '');

            wp_send_json_success([
                'message'        => __('Clave verificada. La API no devolvió ID de cuenta aún. Vincula tu WhatsApp y vuelve a intentar.', 'whatsboost'),
                'account'        => '',
                'definedAccount' => $defined_account,
                'matchesAccount' => false,
                'apiHttpCode'    => $code,
                'apiUrl'         => $url,
                'waAccounts'     => $wa_accounts_list,
            ]);
        }

        // Si seguimos sin cuenta y no hay señal de suscripción OK, tratamos como clave inválida
        if ($account_from_api === '') {
            // Sin señales de suscripción OK: persistir clave y limpiar estado como inválida
            update_option('whatsboost_license_key', $key, false);
            update_option('whatsboost_license_verified', 0, false);
            update_option('whatsboost_account', '', false);
            wp_send_json_error([
                'message' => __('Clave de licencia no válida. Inténtalo de nuevo.', 'whatsboost'),
                'status'  => $code,
            ], 400);
        }

        // Guardar estado verificado y persistir clave y cuenta
        update_option('whatsboost_license_key', $key, false);
        update_option('whatsboost_license_verified', 1, false);

        // Elegir cuenta a usar: respetar selección previa si existe; si no, preferir primera conectada; si no, la primera
        $current_saved = (string) get_option('whatsboost_account', '');
        $chosen_account = $account_from_api;
        if ($current_saved !== '') {
            $chosen_account = $current_saved;
        } else {
            if ($chosen_account === '') {
                foreach ($wa_accounts_list as $it) {
                    $st = isset($it['status']) ? strtolower((string)$it['status']) : '';
                    if (!empty($it['id']) && $st === 'connected') { $chosen_account = (string)$it['id']; break; }
                }
                if ($chosen_account === '' && !empty($wa_accounts_list) && !empty($wa_accounts_list[0]['id'])) {
                    $chosen_account = (string)$wa_accounts_list[0]['id'];
                }
            }
        }
        if ($chosen_account !== '') {
            update_option('whatsboost_account', $chosen_account, false);
        }
        delete_transient('whatsboost_account_check_ts');

        $defined_account = (string) get_option('whatsboost_account', '');
        $matches = ($defined_account !== '' && $account_from_api !== '' && hash_equals($defined_account, $account_from_api));

        wp_send_json_success([
            'message'        => $message,
            'account'        => $account_from_api,
            'definedAccount' => $defined_account,
            'matchesAccount' => $matches,
            'apiHttpCode'    => $code,
            'apiUrl'         => $url,
            'waAccounts'     => $wa_accounts_list,
        ]);

    }

    public function ajax_get_accounts(): void {
        check_ajax_referer('whatsboost_verify', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('No tienes permisos suficientes para realizar esta acción.', 'whatsboost')], 403);
        }
        $key = trim((string) get_option('whatsboost_license_key', ''));
        if ($key === '') {
            wp_send_json_error(['message' => __('No hay clave guardada.', 'whatsboost')], 400);
        }
        $wa_accounts_endpoint = apply_filters('whatsboost_accounts_endpoint', 'https://whatsboost.net/api/get/wa.accounts', $key);
        $wa_url = add_query_arg(['secret' => $key, 'limit' => 100, 'page' => 1], $wa_accounts_endpoint);
        $wa_res = wp_remote_get($wa_url, [ 'headers' => [ 'Accept' => 'application/json', 'Content-Type' => 'application/json' ], 'timeout' => 20 ]);
        if (is_wp_error($wa_res)) {
            wp_send_json_error(['message' => __('Error de red al consultar cuentas.', 'whatsboost'), 'error' => $wa_res->get_error_message()], 500);
        }
        $code = (int) wp_remote_retrieve_response_code($wa_res);
        if (!($code >= 200 && $code < 300)) {
            wp_send_json_error(['message' => __('No se pudo obtener la lista de cuentas.', 'whatsboost'), 'status' => $code], $code);
        }
        $wa_body = wp_remote_retrieve_body($wa_res);
        $wa_json = json_decode($wa_body, true);
        $list = [];
        if (json_last_error() === JSON_ERROR_NONE && is_array($wa_json)) {
            $data = isset($wa_json['data']) && is_array($wa_json['data']) ? $wa_json['data'] : [];
            foreach ($data as $row) {
                if (!is_array($row)) { continue; }
                $id = '';
                foreach (['unique','account','account_id','accountId'] as $k) {
                    if (isset($row[$k]) && is_string($row[$k]) && trim($row[$k]) !== '') { $id = trim((string)$row[$k]); break; }
                }
                $phone = '';
                foreach (['phone','mobile','msisdn','telefono'] as $k) {
                    if (isset($row[$k]) && (is_string($row[$k]) || is_numeric($row[$k]))) { $phone = trim((string)$row[$k]); break; }
                }
                $status = '';
                if (isset($row['status'])) { $status = is_string($row['status']) ? trim($row['status']) : ($row['status'] ? 'connected' : 'disconnected'); }
                elseif (isset($row['state'])) { $status = is_string($row['state']) ? trim($row['state']) : ''; }
                elseif (isset($row['connected'])) { $status = $row['connected'] ? 'connected' : 'disconnected'; }
                $list[] = [ 'id' => $id, 'phone' => $phone, 'status' => $status ];
            }
        }
        $selected = (string) get_option('whatsboost_account', '');
        wp_send_json_success(['waAccounts' => $list, 'selected' => $selected]);
    }

    public function ajax_set_account(): void {
        check_ajax_referer('whatsboost_verify', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('No tienes permisos suficientes para realizar esta acción.', 'whatsboost')], 403);
        }
        $id = isset($_POST['id']) ? sanitize_text_field((string)$_POST['id']) : '';
        if ($id === '') {
            // También permitimos limpiar la selección
            update_option('whatsboost_account', '', false);
            wp_send_json_success(['selected' => '']);
        }
        // Guardar directamente; en una mejora se podría validar contra la lista
        update_option('whatsboost_account', $id, false);
        delete_transient('whatsboost_account_check_ts');
        wp_send_json_success(['selected' => $id, 'message' => __('Número seleccionado guardado.', 'whatsboost')]);
    }

    /* ========= AÑADIDOS PARA EL ENLACE "DOCUMENTACIÓN" ========= */

    /**
     * Enlace "Documentación" en la meta de la fila del plugin
     */
    public function add_docs_meta_link($links, $file) {
        if ($file === $this->plugin_basename) {
            $links[] = '<a href="' . esc_url(admin_url('admin.php?page=whatsboost-docs')) . '">' . esc_html__('Documentación', 'whatsboost') . '</a>';
        }
        return $links;
    }

    /**
     * Enlace "Documentación" en los action links ("Ajustes | Desactivar")
     */
    public function add_action_link_docs($links) {
        $url = admin_url('admin.php?page=whatsboost-docs');
        array_unshift($links, '<a href="' . esc_url($url) . '">' . esc_html__('Documentación', 'whatsboost') . '</a>');
        return $links;
    }
}

/**
 * Devuelve un valor de un array asociativo usando clave con puntos (p. ej. subscription.account)
 */
function whatsboost_array_get_dot($array, $path, $default = null) {
    if (!is_array($array)) { return $default; }
    $keys = explode('.', (string)$path);
    $val = $array;
    foreach ($keys as $k) {
        if (is_array($val) && array_key_exists($k, $val)) {
            $val = $val[$k];
        } else {
            return $default;
        }
    }
    return $val;
}


/**
 * Busca recursivamente en un array/objeto un identificador que parezca una clave HEX larga (32-64 chars).
 * Devuelve la primera coincidencia encontrada o cadena vacía si no hay.
 */
function whatsboost_find_account_like($data) {
    $queue = [$data];
    while (!empty($queue)) {
        $current = array_shift($queue);
        if (is_array($current)) {
            foreach ($current as $v) {
                if (is_array($v)) {
                    $queue[] = $v;
                } elseif (is_string($v)) {
                    $s = trim($v);
                    if ($s !== '' && preg_match('/^[A-Fa-f0-9]{32,64}$/', $s)) {
                        return $s;
                    }
                }
            }
        } elseif (is_string($current)) {
            $s = trim($current);
            if ($s !== '' && preg_match('/^[A-Fa-f0-9]{32,64}$/', $s)) {
                return $s;
            }
        }
    }
    return '';
}
