<?php
namespace WhatsBoost\Woocommerse;

if (!defined('ABSPATH')) { exit; }

/**
 * Encola intl-tel-input en el checkout y guarda el E.164 enviado por el front (Opción A).
 * Se carga solo en la página de checkout.
 */
add_action('wp_enqueue_scripts', __NAMESPACE__ . '\\enqueue_checkout_phone_assets');
function enqueue_checkout_phone_assets(): void {
    if (!function_exists('is_checkout') || !is_checkout()) { return; }

    // CSS/JS desde CDN
    wp_enqueue_style(
        'intl-tel-input',
        'https://cdn.jsdelivr.net/npm/intl-tel-input@19.5.8/build/css/intlTelInput.css',
        [],
        null
    );
    wp_enqueue_script(
        'intl-tel-input',
        'https://cdn.jsdelivr.net/npm/intl-tel-input@19.5.8/build/js/intlTelInput.min.js',
        [],
        null,
        true
    );

    // Script de integración con los campos de WooCommerce
    $inline = "document.addEventListener('DOMContentLoaded',function(){
      var c = document.querySelector('#billing_country');
      var p = document.querySelector('#billing_phone');
      if(!c || !p) return;
      var f = p.form || document.querySelector('form.checkout') || document.querySelector('form');
      if(!f) return;
      var hidden = document.querySelector('#billing_phone_e164');
      if (!hidden) {
        hidden = document.createElement('input');
        hidden.type = 'hidden';
        hidden.name = 'billing_phone_e164';
        hidden.id = 'billing_phone_e164';
        f.appendChild(hidden);
      }
      var iti = window.intlTelInput(p, {
        initialCountry: (c.value||'ES').toLowerCase(),
        separateDialCode: true,
        nationalMode: true,
        utilsScript: 'https://cdn.jsdelivr.net/npm/intl-tel-input@19.5.8/build/js/utils.js'
      });
      c.addEventListener('change', function(){ iti.setCountry((c.value||'').toLowerCase()); });
      f.addEventListener('submit', function(){ try{ hidden.value = iti.getNumber(); }catch(e){} });
    });";
    wp_add_inline_script('intl-tel-input', $inline);
}

/**
 * Guarda el meta _billing_phone_e164 cuando llega desde el checkout (front Opción A).
 */
add_action('woocommerce_checkout_update_order_meta', __NAMESPACE__ . '\\save_billing_phone_e164', 10, 2);
function save_billing_phone_e164($order_id, $data): void {
    $e164 = isset($_POST['billing_phone_e164']) ? sanitize_text_field((string) $_POST['billing_phone_e164']) : '';
    if ($e164 && preg_match('/^\+\d{7,15}$/', $e164)) {
        update_post_meta($order_id, '_billing_phone_e164', $e164);
    }
}

/**
 * Inicializa los hooks específicos de WooCommerce para el plugin.
 */
function init(): void {
    // Solo seguir si WooCommerce está disponible
    if (!function_exists('wc_get_order')) { return; }

    // Respetar la condición de licencia: solo enganchamos si la clave está verificada
    $license = trim((string) get_option('whatsboost_license_key', ''));
    $verified = (int) get_option('whatsboost_license_verified', 0);
    if ($license === '' || $verified !== 1) { return; }

    // Envío clásico en la página de agradecimiento
    add_action('woocommerce_thankyou', __NAMESPACE__ . '\\send_whatsapp_on_order', 20, 1);

    // NUEVO: envío al cambiar el estado de pedido
    add_action('woocommerce_order_status_changed', __NAMESPACE__ . '\\maybe_send_status_message', 10, 3);
}

/**
 * Envía un WhatsApp de agradecimiento cuando se genera un pedido.
 * Respeta la condición de que exista una clave válida (opción whatsboost_license_key).
 */
function send_whatsapp_on_order($order_id): void {
    try {
        $license = trim((string) get_option('whatsboost_license_key', ''));
        if ($license === '') { return; }
        if (empty($order_id)) { return; }
        if (!function_exists('wc_get_order')) { return; }

        $order = wc_get_order($order_id);
        if (!$order) { return; }

        // Teléfono del cliente priorizando el meta _billing_phone_e164 del front (Opción A)
        $phone = get_recipient_phone_e164($order);
        if ($phone === '') { return; }

        // Construir mensaje
        $store = get_bloginfo('name');
        $order_number = method_exists($order, 'get_order_number') ? $order->get_order_number() : (string) $order_id;
        $message = sprintf(
            '¡Gracias por tu compra en %1$s! Tu pedido #%2$s ha sido recibido y ya está en proceso. Valoramos mucho tu confianza. Si necesitas ayuda, responde a este WhatsApp y te atenderemos enseguida. Como agradecimiento, te regalamos un 5%% de descuento en tu próxima compra con el código VUELVE5. ¡Esperamos verte pronto! 🛍️💚',
            $store,
            $order_number
        );

        send_whatsapp($phone, $message, $license);
    } catch (\Throwable $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[WhatsBoost] Error al preparar envío de WhatsApp: ' . $e->getMessage());
        }
    }
}

function send_whatsapp(string $to, string $message, string $license): void {
    $endpoint = apply_filters('whatsboost_whatsapp_endpoint', 'https://whatsboost.net/api/send/whatsapp');

    $body = [
        'secret'    => $license, // mantenemos por compatibilidad con servidores que aún lo acepten en el body
        'account'   => get_option('whatsboost_account', ''),
        'recipient' => $to,
        'type'      => 'text',
        'message'   => $message,
    ];

    $args = [
        'timeout' => 20,
        'headers' => [
            'Accept'        => 'application/json',
            // Algunos endpoints exigen Authorization: Bearer <secret>
            'Authorization' => 'Bearer ' . $license,
        ],
        'body'    => $body,
    ];

    $response = wp_remote_post($endpoint, $args);
    if (is_wp_error($response)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[WhatsBoost] Error HTTP WhatsApp: ' . $response->get_error_message());
        }
        return;
    }

    $code = (int) wp_remote_retrieve_response_code($response);
    if ($code < 200 || $code >= 300) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[WhatsBoost] Envío WhatsApp no exitoso. Código: ' . $code . ' - Respuesta: ' . wp_remote_retrieve_body($response));
        }
    }
}

/**
 * Normaliza el teléfono a E.164 usando país y región del pedido/tienda.
 * - Acepta números ya con prefijo + o con 00.
 * - Si no hay prefijo internacional, se usa el código del país del pedido
 *   (facturación, luego envío) o el país por defecto de WooCommerce.
 */
function normalize_phone(string $raw, $order): string {
    $raw = trim((string) $raw);
    if ($raw === '') { return ''; }

    // Mantener solo dígitos y signos + y 0 para comprobar 00
    $clean = preg_replace('/[^0-9+]/', '', $raw);
    if ($clean === null) { $clean = ''; }
    if ($clean === '') { return ''; }

    // Convertir prefijo 00 a +
    if (strpos($clean, '00') === 0) {
        $clean = '+' . substr($clean, 2);
    }

    // Si ya viene en formato internacional
    if ($clean[0] === '+') {
        return $clean; // confiar en que el usuario ha puesto el prefijo correcto
    }

    // Determinar país desde el pedido o la tienda
    $iso2 = '';
    if (is_object($order)) {
        $iso2 = method_exists($order, 'get_billing_country') ? (string) $order->get_billing_country() : '';
        if ($iso2 === '') {
            $iso2 = method_exists($order, 'get_shipping_country') ? (string) $order->get_shipping_country() : '';
        }
    }
    if ($iso2 === '') {
        $default = (string) get_option('woocommerce_default_country', '');
        if ($default !== '') {
            $parts = explode(':', $default);
            $iso2 = strtoupper(trim($parts[0] ?? ''));
        }
    }
    $iso2 = strtoupper($iso2);

    $code = get_calling_code($iso2);
    if ($code === '') {
        // Fallback razonable: España (+34)
        $code = '34';
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[WhatsBoost] ISO2 desconocido para teléfono, usando +34. ISO2=' . $iso2);
        }
    }

    // Quitar ceros iniciales del número nacional (común en Europa)
    $nsn = ltrim($clean, '0');
    if ($nsn === '') { return ''; }

    // Si ya empieza con el código del país (sin +), evita duplicarlo
    if ($code !== '' && strpos($nsn, $code) === 0) {
        return '+' . $nsn;
    }
    return '+' . $code . $nsn;
}

/**
 * Obtiene el teléfono del destinatario priorizando el meta guardado en E.164
 * por el front (Opción A). Si no existe, intenta usar billing_phone en E.164
 * y por último normaliza con normalize_phone().
 */
function get_recipient_phone_e164($order): string {
    if (!is_object($order)) { return ''; }

    // 1) Meta normalizado desde el front
    if (method_exists($order, 'get_meta')) {
        $saved = (string) $order->get_meta('_billing_phone_e164', true);
        if ($saved && preg_match('/^\+\d{7,15}$/', $saved)) {
            return $saved;
        }
    }

    // 2) Si el billing_phone ya viene en E.164, úsalo
    $raw = method_exists($order, 'get_billing_phone') ? (string) $order->get_billing_phone() : '';
    if ($raw && preg_match('/^\+\d{7,15}$/', $raw)) {
        return $raw;
    }

    // 3) Fallback: normalizar
    $norm = normalize_phone($raw, $order);
    return $norm;
}

/**
 * Devuelve el código telefónico internacional para un país ISO2.
 */
function get_calling_code(string $iso2): string {
    $map = [
        'ES'=>'34','AR'=>'54','MX'=>'52','US'=>'1','CA'=>'1','GB'=>'44','UK'=>'44','FR'=>'33','DE'=>'49','IT'=>'39','PT'=>'351','BR'=>'55','CL'=>'56','CO'=>'57','PE'=>'51','VE'=>'58','UY'=>'598','PY'=>'595','BO'=>'591','EC'=>'593','CR'=>'506','PA'=>'507','DO'=>'1','PR'=>'1','GT'=>'502','HN'=>'504','SV'=>'503','NI'=>'505','CU'=>'53','AU'=>'61','NZ'=>'64','IE'=>'353','NL'=>'31','BE'=>'32','CH'=>'41','AT'=>'43','SE'=>'46','NO'=>'47','DK'=>'45','FI'=>'358','PL'=>'48','CZ'=>'420','SK'=>'421','HU'=>'36','RO'=>'40','BG'=>'359','GR'=>'30','TR'=>'90','RU'=>'7','UA'=>'380','IN'=>'91','PK'=>'92','BD'=>'880','CN'=>'86','JP'=>'81','KR'=>'82','SA'=>'966','AE'=>'971','MA'=>'212','EG'=>'20','ZA'=>'27','IL'=>'972','IR'=>'98','HK'=>'852','SG'=>'65','MY'=>'60','TH'=>'66','PH'=>'63','ID'=>'62','VN'=>'84','KH'=>'855','LA'=>'856','TW'=>'886','IS'=>'354','LT'=>'370','LV'=>'371','EE'=>'372','SI'=>'386','HR'=>'385','BA'=>'387','RS'=>'381','MK'=>'389','AL'=>'355','GE'=>'995','AM'=>'374','AZ'=>'994','KZ'=>'7','BY'=>'375','MD'=>'373','QA'=>'974','KW'=>'965','BH'=>'973','OM'=>'968','JO'=>'962','LB'=>'961','IQ'=>'964','SY'=>'963','YE'=>'967','KE'=>'254','NG'=>'234','GH'=>'233','TZ'=>'255','UG'=>'256','ET'=>'251','TN'=>'216','DZ'=>'213','AO'=>'244','CM'=>'237','CI'=>'225','SN'=>'221','ML'=>'223','NE'=>'227','RW'=>'250','SD'=>'249','MR'=>'222','GM'=>'220','GQ'=>'240','GA'=>'241','MZ'=>'258','ZM'=>'260','ZW'=>'263','BW'=>'267','NA'=>'264','MU'=>'230','RE'=>'262'
    ];
    $iso2 = strtoupper(trim($iso2));
    return $map[$iso2] ?? '';
}


/**
 * Reemplaza variables del mensaje con datos del pedido.
 * Variables soportadas (inglés y español):
 * {order_number}, {first_name}, {last_name}, {billing_phone}, {total}, {currency}, {order_date}, {status}, {store_name}, {site_url}, {order_url}
 * {nombre}, {tienda}, {link}, {producto_pedido}, {cantidad}, {precio}, {plazo}, {tracking}
 *
 * @param string $message
 * @param \WC_Order $order
 * @return string
 */
function replace_message_placeholders(string $message, $order): string {
    if (!is_object($order)) { return $message; }

    try {
        $order_number = method_exists($order, 'get_order_number') ? (string) $order->get_order_number() : (string) $order->get_id();
        $first_name   = method_exists($order, 'get_billing_first_name') ? (string) $order->get_billing_first_name() : '';
        $last_name    = method_exists($order, 'get_billing_last_name') ? (string) $order->get_billing_last_name() : '';
        $phone        = method_exists($order, 'get_billing_phone') ? (string) $order->get_billing_phone() : '';
        $total_raw    = method_exists($order, 'get_total') ? (string) $order->get_total() : '';
        $currency     = method_exists($order, 'get_currency') ? (string) $order->get_currency() : '';
        $date_created = method_exists($order, 'get_date_created') ? $order->get_date_created() : null;
        $date_str     = $date_created ? $date_created->date_i18n(get_option('date_format') . ' ' . get_option('time_format')) : '';
        $status       = method_exists($order, 'get_status') ? (string) $order->get_status() : '';
        $store_name   = get_bloginfo('name');
        $site_url     = site_url('/');
        $order_url    = method_exists($order, 'get_view_order_url') ? (string) $order->get_view_order_url() : '';

        // Formato de precio en texto plano
        $price_formatted = $total_raw;
        if (function_exists('wc_price')) {
            $price_formatted = wp_strip_all_tags(wc_price((float) $total_raw, ['currency' => $currency]));
        } elseif ($total_raw !== '') {
            $price_formatted = trim($total_raw . ($currency !== '' ? ' ' . $currency : ''));
        }

        // Primer producto y su cantidad
        $first_item_name = '';
        $first_item_qty  = '';
        if (method_exists($order, 'get_items')) {
            $items = $order->get_items();
            if (!empty($items) && is_iterable($items)) {
                foreach ($items as $item) {
                    if (is_object($item)) {
                        $first_item_name = method_exists($item, 'get_name') ? (string) $item->get_name() : '';
                        $first_item_qty  = method_exists($item, 'get_quantity') ? (string) $item->get_quantity() : '';
                        break;
                    }
                }
            }
        }

        // Método de envío como proxy de {plazo}
        $plazo = method_exists($order, 'get_shipping_method') ? (string) $order->get_shipping_method() : '';

        // Tracking si existe en metadatos comunes
        $tracking = '';
        if (method_exists($order, 'get_meta')) {
            $meta_keys = ['_tracking_number', 'tracking_number', '_wc_shipment_tracking_items'];
            foreach ($meta_keys as $mk) {
                $val = $order->get_meta($mk, true);
                if (!empty($val)) {
                    if (is_array($val)) {
                        $first = reset($val);
                        if (is_array($first)) {
                            $tracking = (string) ($first['tracking_link'] ?? ($first['tracking_number'] ?? ''));
                        } else {
                            $tracking = (string) $first;
                        }
                    } else {
                        $tracking = (string) $val;
                    }
                    if ($tracking !== '') { break; }
                }
            }
        }

        $replacements = [
            // Inglés (existente)
            '{order_number}' => $order_number,
            '{first_name}'   => $first_name,
            '{last_name}'    => $last_name,
            '{billing_phone}'=> $phone,
            '{total}'        => $total_raw,
            '{currency}'     => $currency,
            '{order_date}'   => $date_str,
            '{status}'       => $status,
            '{store_name}'   => $store_name,
            '{site_url}'     => $site_url,
            '{order_url}'    => $order_url,

            // Español (solicitado)
            '{nombre}'           => $first_name,
            '{apellido}'         => $last_name,
            '{tienda}'           => $store_name,
            '{link}'             => $order_url,
            '{producto_pedido}'  => $first_item_name,
            '{cantidad}'         => $first_item_qty,
            '{precio}'           => $price_formatted,
            '{plazo}'            => $plazo,
            '{tracking}'         => $tracking,
        ];

        return strtr($message, $replacements);
    } catch (\Throwable $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[WhatsBoost] Error en replace_message_placeholders: ' . $e->getMessage());
        }
        return $message;
    }
}


/**
 * Envía el mensaje configurado cuando un pedido cambia a un estado habilitado.
 *
 * @param int    $order_id
 * @param string $old_status  p.ej. 'pending'
 * @param string $new_status  p.ej. 'completed'
 */
function maybe_send_status_message($order_id, string $old_status, string $new_status): void {
    try {
        $license = trim((string) get_option('whatsboost_license_key', ''));
        if ($license === '') { return; }
        if (empty($order_id)) { return; }
        if (!function_exists('wc_get_order')) { return; }

        $order = wc_get_order($order_id);
        if (!$order) { return; }

        // Clave tal como se guarda en los ajustes (wc-<estado>)
        $status_key = 'wc-' . sanitize_key($new_status); // p.ej. 'wc-completed'

        // Leer opciones guardadas
        $enabled = get_option('whatsboost_status_enabled', []);
        $texts   = get_option('whatsboost_status_text', []);
        if (!is_array($enabled)) { $enabled = []; }
        if (!is_array($texts)) { $texts = []; }

        // ¿Está habilitado este estado?
        $is_enabled = !empty($enabled[$status_key]);
        if (!$is_enabled) { return; }

        // Mensaje configurado para este estado
        $message = isset($texts[$status_key]) ? trim((string) $texts[$status_key]) : '';
        if ($message === '') { return; }

        // Reemplazar placeholders si existe la función
        if (function_exists(__NAMESPACE__ . '\\replace_message_placeholders')) {
            $message = replace_message_placeholders($message, $order);
        }

        // Teléfono del cliente priorizando el meta E.164 del front
        $phone = get_recipient_phone_e164($order);
        if ($phone === '') { return; }

        // Enviar
        send_whatsapp($phone, $message, $license);
    } catch (\Throwable $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[whatsboost] Error en maybe_send_status_message: ' . $e->getMessage());
        }
    }
}
